#!/usr/bin/env python3
import sys
from os.path import basename
import struct
from datetime import datetime
import argparse
from PIL import Image
import io
import json
import os

# Set Image Info
# Image dimensions
img_width = 256
img_height = 192

# Screen Shot Format
ss_image_top = 0x02a4
ss_image_len = img_width * img_height
ss_file_len = 99360

# BMP Format (FullColor)
bmp_image_top = 0x0036
bmp_image_len = img_width * img_height * 3
bmp_image = bytearray([0] * bmp_image_len)
bmp_file_len = 147512

# BMP Headers
bmp_headers = bytearray.fromhex(
    "424D3840020000000000360000002800000000010000C0000000010018000000000000000000120B0000120B00000000000000000000"
)

# Other configurations
base_domain = "http://ragol.net/"
upload_path = ""
full_page_path = f"{base_domain}"

timestamp = datetime.now().strftime("%Y%m%d%H%M%S")

def make_pathname(vmi_filename):
    path = "/".join(vmi_filename.split("/")[:-1])
    if not path:
        path = "."
    return path

def get_extension(filename):
    return filename.split(".")[-1].lower()

def write_jpeg(bmp_data, image_filename):
    base_filename, ext = os.path.splitext(image_filename)
    num = 1
    while os.path.exists(image_filename):
        image_filename = f"{base_filename}_{num}{ext}"
        num += 1

    with io.BytesIO(bmp_data) as bmp_io:
        bmp_image = Image.open(bmp_io)
        bmp_image.save(image_filename, "JPEG")

def write_png(bmp_data, image_filename):
    base_filename, ext = os.path.splitext(image_filename)
    num = 1
    while os.path.exists(image_filename):
        image_filename = f"{base_filename}_{num}{ext}"
        num += 1

    with io.BytesIO(bmp_data) as bmp_io:
        bmp_image = Image.open(bmp_io)
        bmp_image.save(image_filename, "PNG")

def swap_in_place(data):
    swapped_data = bytearray(data)
    for x in range(0, len(swapped_data), 4):
        if x + 3 < len(swapped_data):
            swapped_data[x], swapped_data[x + 1], swapped_data[x + 2], swapped_data[x + 3] = swapped_data[x + 3], swapped_data[x + 2], swapped_data[x + 1], swapped_data[x]
    return swapped_data

def dec_to_bcd(input):
    return (input // 10) << 4 | (input % 10)


def get_vmi_value(vmi_flat, index):
    return int(str(vmi_flat[index])) if str(vmi_flat[index]).isdigit() else 0

def read_vmi_file(vmi_file):
    with open(vmi_file, "rb") as vmifile:
        return vmifile.read(108)

def read_vms_file(vmsfilename, bytesize):
    with open(vmsfilename, "rb") as vmsfile:
        return vmsfile.read(bytesize)

def create_dci_header(vmi_flat):
    dciheader = bytearray([0] * 0x20)
    vmi_86 = int(vmi_flat[86]) if str(vmi_flat[86]).isdigit() else 0
    dciheader[0x00] = 0xcc if (vmi_86 & 2) == 2 else 0x33
    dciheader[0x01] = 0xff if (vmi_86 & 1) == 1 else 0x00

    for i in range(88, 104):
        dciheader[0x00 + i - 84] = vmi_flat[i]

    dciheader[0x10:0x18] = [
        dec_to_bcd(struct.unpack("<H", vmi_flat[78:80])[0] // 100),
        dec_to_bcd(struct.unpack("<H", vmi_flat[78:80])[0] % 100),
        dec_to_bcd(get_vmi_value(vmi_flat, 80)),
        dec_to_bcd(get_vmi_value(vmi_flat, 81)),
        dec_to_bcd(get_vmi_value(vmi_flat, 82)),
        dec_to_bcd(get_vmi_value(vmi_flat, 83)),
        dec_to_bcd(get_vmi_value(vmi_flat, 84)),
        ((get_vmi_value(vmi_flat, 85)) + 6) % 7
    ]
    return dciheader

def combine_header_vmsdata(dciheader, vmsdata):
    return dciheader + vmsdata

def vmi_file_to_dci_data(vmi_file, path):
    vmi_flat = read_vmi_file(vmi_file)
    lowsize, highsize = struct.unpack_from("<HH", vmi_flat, 104)
    bytesize = (highsize << 16) + lowsize
    vmsfilename = path + "/" + basename(vmi_file)[:-4] + ".VMS"
    vmsdata = read_vms_file(vmsfilename, bytesize)
    vmsdata = swap_in_place(vmsdata)
    dciheader = create_dci_header(vmi_flat)
    dci_data = combine_header_vmsdata(dciheader, vmsdata)
    return dci_data

def dci_data_to_bmp(dci_data):
    buf = bytearray(dci_data[:ss_file_len])
    if len(buf) != ss_file_len or buf[4:16] != b"PSO______IMG":
        sys.exit("Not PSO Screen Shot File.")

    for i in range(ss_file_len // 4):
        if ss_image_top + i * 4 + 4 > len(buf):
            break
        a, b, c, d = struct.unpack("BBBB", buf[ss_image_top + i * 4 : ss_image_top + i * 4 + 4])
        buf[i * 4 : i * 4 + 4] = struct.pack("BBBB", d, c, b, a)

    for y in range(img_height):
        for x in range(img_width):
            lo, hi = struct.unpack("BB", buf[(x + y * img_width) * 2 : (x + y * img_width) * 2 + 2])
            pal = (hi << 8) + lo
            r = (pal >> 8) & 0xf8 if (pal >> 8) else 0
            r += 0x7 if r else 0
            g = (pal >> 3) & 0xfc if (pal >> 3) else 0
            g += 0x3 if g else 0
            b = (pal << 3) & 0xf8 if (pal << 3) else 0
            b += 0x7 if b else 0
            bmp_image[
                (x + (img_height - y - 1) * img_width) * 3 : (x + (img_height - y - 1) * img_width) * 3 + 3
            ] = struct.pack("BBB", b, g, r)

    bmp_data = bmp_headers + bmp_image + bytearray.fromhex("00 00")
    return bmp_data

def dci_file_to_dci_data(input_file):
    with open(input_file, "rb") as ss_in:
        dci_data = bytearray(ss_in.read(ss_file_len))
    return dci_data

def flip_4_byte_pairs(byte_string):
    flipped_pairs = [byte_string[i:i+4][::-1] for i in range(0, len(byte_string), 4)]
    flipped_string = b''.join(flipped_pairs)
    return flipped_string

def dci_get_names(dci_data):
    chunk_size = 24
    start_position = 99080  # Start position after byte 99080
    player_data = dci_data[start_position:start_position + 288]
    player_byte_flip_data = flip_4_byte_pairs(player_data)
    # Split byte_flip_data into chunks of 24 bytes
    player_data_chunks = [player_byte_flip_data[i:i+chunk_size] for i in range(0, len(player_byte_flip_data), chunk_size)]
    cleaned_player_list = [item.replace(b'\x00', b'') for item in player_data_chunks if item.replace(b'\x00', b'')]
    decoded_player_list = [item.decode('utf-8') for item in cleaned_player_list]
    return decoded_player_list

def make_filenames(timestamp):
    base_filename = f"{timestamp}"
    # jpg_filename = f"{base_filename}.jpg"
    png_filename = f"{base_filename}.png"
    html_filename = f"{base_filename}.html"
    num = 1
    while os.path.exists(png_filename) or os.path.exists(html_filename):
        base_filename = f"{timestamp}_{num}"
        # jpg_filename = f"{base_filename}.jpg"
        png_filename = f"{base_filename}.png"
        html_filename = f"{base_filename}.html"
        num += 1
    return png_filename, html_filename

def main():
    parser = argparse.ArgumentParser(description='Process VMI and DCI files to BMP and JPEG.')
    parser.add_argument('input_file', type=str, help='Input file (VMI or DCI)')
    args = parser.parse_args()

    input_file = args.input_file
    ext = get_extension(input_file)
    path = make_pathname(input_file)

    if ext == "vmi":
        print(input_file)
        dci_data = vmi_file_to_dci_data(input_file, path)
        player_names = dci_get_names(dci_data)
    elif ext == "vms": # Convert to VMI incase user put in wrong file
        input_file=input_file[:-4] + ".VMI"
        dci_data = vmi_file_to_dci_data(input_file, path)
        player_names = dci_get_names(dci_data)
    elif ext == "dci":
        dci_data = dci_file_to_dci_data(input_file)
        player_names = dci_get_names(dci_data)
    else:
        print("Unsupported file format.")
        return

    bmp_data = dci_data_to_bmp(dci_data)
    timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
    png_filename, html_filename = make_filenames(timestamp)

    # write_jpeg(bmp_data, jpg_filename)
    write_png(bmp_data, upload_path+png_filename)

    html_content = f"""
    <!DOCTYPE html>
    <html>
    <head>
    <title>Dreamcast Image Data</title>
    </head>
    <body>
    <table align="center" border="1">
        <tr>
            <td style="text-align: center;"><img src="{png_filename}" alt="Screenshot"></td>
        </tr>
        <tr>
            <td>
                <strong>Timestamp:</strong> {timestamp}<br>
                <strong>Nearby Players:</strong><br>
                <ul>{"".join(f'<li>{player}</li>' for player in player_names)}</ul>
            </td>
        </tr>
    </table>
    </body>
    </html>
    """
    with open(upload_path+html_filename, "w") as html_file:
        html_file.write(html_content)
    json_data = {
        "timestamp": timestamp,
        "image": png_filename,
        "page": html_filename,
        "players": player_names
    }
    print(json.dumps(json_data, indent=4))
if __name__ == "__main__":
    main()
